/* main_Z4.c - eDMA Transfer example for S32R274 */
/* Description:  Makes eDMA transfers of data from Source to Destination buffers */
/* Rev 1.0 Aug 10 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2018 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main_Z4.c
* Owner             David Chung
* Version           1.0
* Date              Aug-10-2018
* Classification    General Business Information
* Brief             Tests S32R274 eDMA. Transfers data from one buffer in flash
* 					to another buffer in flash.
********************************************************************************
* Detailed Description:
* Application initializes eDMA and automatically starts transfers of "Hello World"
* from source buffer to destination buffer. Add a watch on TCD0_SourceData and
* TCD0_Destination by entering the variable names in the "Expressions" window of
* Debug view.  Click dropdown menu on TCD0_SourceData and TCD0_Destination to
* reveal contents of the arrays.  TCD0_SourceData will start with "Hello World"
* and TCD0_Destination will start with "\0".  Run until user LED turns on.
* Pause execution and check the buffers again.  TCD0_Destination will change to
* "Hello World".  The UART will print the contents of TCD0_Destination before
* and after eDMA transmission.
*
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB+MPC57xxMB
* MCU:             S32R274
* Terminal:        19200, 8N1, None
* Fsys:            160 MHz
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Connect S32R274RRUEVB to computer through mini-USB.
* 					Connect P8.1 to user LED P7.1
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-10-2018  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */
#include "project.h"
#include "mode_entry.h"
#include "edma.h"
#include "smpu.h"
#include "uart.h"
#include <string.h>

#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

extern void xcptn_xmpl(void);
extern void initTCDs(void);

extern uint8_t TCD0_SourceData[];
extern uint8_t TCD0_Destination[];

void bridges_config(void);
void PIT_Init(void);

void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

/*****************************************************************************/
/* bridges_config                                                            */
/* Description: Configures bridges to provide desired RWX and user/supervisor*/
/*              access and priorites by crossbar masters to crossbar slaves. */
/*****************************************************************************/

void bridges_config (void) {
	/* Enable DMA RWU for peripherals belonging to PBRIDGE_A.
	 * You can see which module belongs to which P-bridge in
	 * the AIPS0 and AIPS1 columns of "Peripheral map" tab within
	 * "S32R274_Memory_map_Rev15.xlsx".
	 *
	 * Enable DMA to write to PBRIDGE_A. Configure the read/write
	 * privilege and priviege level corresponding to DMA 2.
	 * Each MTR/MTW/MPL bit set in the MPRA register corresponds to a master.
	 * Ex. MTR2/MTW2/MPL2 correspond to Master 2. You can check S32R274's
	 * logical bus IDs in the SMPU chapter of the RM, Table 24-1 "Master IDs".
	 * The table shows that eDMA corresponds to Master 2.
	 */
    AIPS_0.MPRA.R |= 0x00700000;   /* Set MTR2, MTW2, and MPL2 to give eDMA R/W privileges to PBridge_A peripherals */
}

void PIT_Init(){
	PIT_0.MCR.R = 0x00000003; //Freeze and disable the module

	/* Configure PIT_0 Ch0 */
	PIT_0.TIMER[0].LDVAL.R = 20000000; //Configure timeout at .5s. PIT runs off PBRIDGE_x_CLK (40 MHz). 20m ticks = 0.5s
	PIT_0.TIMER[0].TCTRL.R = 0x00000001; //Enable Ch0

	PIT_0.MCR.R = 0x00000000; //Reenable and unfreeze the PIT
}

__attribute__ ((section(".text")))
/************************************ Main ***********************************/
int main(void)
{
	  volatile uint32_t i = 0;              /* Dummy idle counter */
	  uint8_t intro[] = {"\n\rWelcome to the S32R274RRUEVB eDMA code example!\n\r"};
	  uint8_t BeforeMessage[] = {"\n\rBefore eDMA transmission, "};
	  uint8_t AfterMessage[] = {"\n\rAfter eDMA transmission, "};
	  uint8_t SourceMessage[] = {"TCD0_SourceData reads "};
	  uint8_t EmptyMessage[] = {"TCD0_Destination is empty."};
	  uint8_t DestinationMessage[] = {"TCD0_Destination reads "};
	  uint8_t EndMessage[] = {"\n\rEnd of S32R274 eDMA test.\n\r"};
	  uint8_t period[] = {"."};
	  uint8_t newline[] = {"\n\r"};

	  xcptn_xmpl ();              /* Configure and Enable Interrupts */

	  bridges_config();       /* Enable R/W to peripherals by DMA */
	  smpu_config();          /* Cache inhibit a RAM region for shared data */
	  system160mhz();         /* sysclk=160MHz, dividers configured, mode trans*/

	  SIUL2.MSCR[0].B.OBE = 1;		/* Enable PA0 as output */
	  SIUL2.GPDO[0].B.PDO_4n = 1;	/* Turn off LED. Set high */

	  init_edma_channel_arbitration(); /* Initialze arbitration among channels */
	  initTCDs();             /* Initialize DMA Transfer Control Descriptors */

	  LINFlexD_1_Init();		/* Initialize LINFlexD_1 for UART transmission */

	  PIT_Init();				/* Configure the PIT to time the DMA transfers to LED */

	  /* Print UART messages */
	  TransmitData((const char*)intro,(uint32_t)strlen((const char*)intro));
	  TransmitData((const char*)BeforeMessage,(uint32_t)strlen((const char*)BeforeMessage));
	  TransmitData((const char*)SourceMessage,(uint32_t)strlen((const char*)SourceMessage));
	  TransmitData((const char*)TCD0_SourceData,(uint32_t)strlen((const char*)TCD0_SourceData));
	  TransmitData((const char*)newline,(uint32_t)strlen((const char*)newline));

	  if(strlen((const char*)TCD0_Destination) == 0){
		  TransmitData((const char*)EmptyMessage,(uint32_t)strlen((const char*)EmptyMessage));
	  }else{
		  TransmitData((const char*)DestinationMessage,(uint32_t)strlen((const char*)DestinationMessage));
		  TransmitData((const char*)TCD0_Destination,(uint32_t)13);
		  TransmitData((const char*)period,(uint32_t)strlen((const char*)period));
		  TransmitData((const char*)newline,(uint32_t)strlen((const char*)newline));
	  }


	  DMA.SERQ.R = 0;        /* Enable EDMA channel 0 */
	                          /* Initiate DMA service using software activation: */

	  DMA.SSRT.R = 0;        /* Set chan 0 START bit to initiate 1st minor loop */
	  while (DMA.TCD[0].CITER.ELINKNO.B.CITER != 1) {
	                          /* while CITER != 1 (not on last minor loop), */
	                          /* wait for START=0 and ACTIVE=0 */
	    while ((DMA.TCD[0].CSR.B.START == 1) | (DMA.TCD[0].CSR.B.ACTIVE == 1)) {}
	    DMA.SSRT.R = 0;     /* Set chan 0 START bit again for next minor loop */
	  }

	  DMA.SSRT.R = 1;        /* Set chan 1 START bit to initiate 1st minor loop */
	  while (DMA.TCD[1].CITER.ELINKNO.B.CITER != 1) {
	                          /* while CITER != 1 (not on last minor loop), */
	                          /* wait for START=0 and ACTIVE=0, and PIT channel to timeout */
	    while ((DMA.TCD[1].CSR.B.START == 1) | (DMA.TCD[1].CSR.B.ACTIVE == 1) | (PIT_0.TIMER[0].TFLG.R == 0)) {}
	    DMA.SSRT.R = 1;     /* Set chan 0 START bit again for next minor loop */
	    PIT_0.TIMER[0].TFLG.R = 0x00000001; //Clear the PIT flag. W1C
	  }

	  /* Transmit UART messages */
	  TransmitData((const char*)AfterMessage,(uint32_t)strlen((const char*)AfterMessage));
	  TransmitData((const char*)SourceMessage,(uint32_t)strlen((const char*)SourceMessage));
	  TransmitData((const char*)TCD0_SourceData,(uint32_t)strlen((const char*)TCD0_SourceData));
	  TransmitData((const char*)newline,(uint32_t)strlen((const char*)newline));

	  if(strlen((const char*)TCD0_Destination) == 0){
		  TransmitData((const char*)EmptyMessage,(uint32_t)strlen((const char*)EmptyMessage));
	  }else{
		  TransmitData((const char*)DestinationMessage,(uint32_t)strlen((const char*)DestinationMessage));
		  TransmitData((const char*)TCD0_Destination,(uint32_t)13);
		  TransmitData((const char*)period,(uint32_t)strlen((const char*)period));
		  TransmitData((const char*)newline,(uint32_t)strlen((const char*)newline));
	  }

	  TransmitData((const char*)EndMessage,(uint32_t)strlen((const char*)EndMessage));

	  while (1) {i++;}
	  return 0;
}


